#include <stdio.h>
#include <string.h>
#include <math.h>
#include "psprubik.h"
#include <stdlib.h>

#define ASSERT(e) if(e); \
    else {printf("Assertion error in line %d\n",__LINE__);return &solverContext;}

#define face(x) ((x)>>3)           // The face number is also the face colour
#define is_corner(x) ((x)&1)
#define is_edge(x) (!is_corner(x))
#define next(x) (((x)&56)+(((x)+1)&7))  // The next sticker clockwise
#define prev(x) (((x)&56)+(((x)-1)&7))  // The next sticker counter-clockwise
#define adj(x) (adjoin_table[(x)])

#define UNDOMAX 4096

/* Here is the layout of the cube in memory:
          43 44 45
          42    46
          41 40 47
    
          11 12 13
          10    14
           9  8 15

39 32 33   7  0  1  23 16 17
38    34   6     2  22    18
37 36 35   5  4  3  21 20 19

          31 24 25
          30    26
          29 28 27
*/

int adjoin_table[48] = {
    8,15,22,21,24,31,34,33,0,7,32,39,40,47,16,23,14,13,46,45,26,25,2,1,4,3,20,19,44,43,36,35,10,9,6,5,30,29,42,41,12,11,38,37,28,27,18,17
};

void init_cube( int cube[])
{
    int i;
    for( i=0; i<48; i++) cube[i] = face(i);
}

struct Face positions[48]={
	{ 0,1,1, 0,0,1 },	// 0
	{ 1,1,1, 0,0,1 },	// 1
	{ 1,0,1, 0,0,1 },	// 2
	{ 1,-1,1, 0,0,1 },	// 3
	{ 0,-1,1, 0,0,1 },	// 4
	{ -1,-1,1, 0,0,1 },	// 5
	{ -1,0,1, 0,0,1 },	// 6
	{ -1,1,1, 0,0,1 },	// 7
	
	{ 0,1,1, 0,1,0 },	// 8
	{ -1,1,1, 0,1,0 },	// 9
	{ -1,1,0, 0,1,0 },	// 10
	{ -1,1,-1, 0,1,0 },	// 11
	{ 0,1,-1, 0,1,0 },	// 12
	{ 1,1,-1, 0,1,0 },	// 13
	{ 1,1,0, 0,1,0 },	// 14
	{ 1,1,1, 0,1,0 },	// 15
	
	{ 1,1,0, 1,0,0 },	// 16
	{ 1,1,-1, 1,0,0 },	// 17
	{ 1,0,-1, 1,0,0 },	// 18
	{ 1,-1,-1, 1,0,0 },	// 19
	{ 1,-1,0, 1,0,0 },	// 20
	{ 1,-1,1, 1,0,0 },	// 21
	{ 1,0,1, 1,0,0 },	// 22
	{ 1,1,1, 1,0,0 },	// 23
	
	{ 0,-1,1, 0,-1,0 },	// 24
	{ 1,-1,1, 0,-1,0 },	// 25
	{ 1,-1,0, 0,-1,0 },	// 26
	{ 1,-1,-1, 0,-1,0 },	// 27
	{ 0,-1,-1, 0,-1,0 },	// 28
	{ -1,-1,-1, 0,-1,0 },	// 29
	{ -1,-1,0, 0,-1,0 },	// 30
	{ -1,-1,1, 0,-1,0 },	// 31
	
	{ -1,1,0, -1,0,0 },	// 32
	{ -1,1,1, -1,0,0 },	// 33
	{ -1,0,1, -1,0,0 },	// 34
	{ -1,-1,1, -1,0,0 },	// 35
	{ -1,-1,0, -1,0,0 },	// 36
	{ -1,-1,-1, -1,0,0 },	// 37
	{ -1,0,-1, -1,0,0 },	// 38
	{ -1,1,-1, -1,0,0 },	// 39
	
	{ 0,1,-1, 0,0,-1 },	// 40
	{ -1,1,-1, 0,0,-1 },	// 41
	{ -1,0,-1, 0,0,-1 },	// 42
	{ -1,-1,-1, 0,0,-1 },	// 43
	{ 0,-1,-1, 0,0,-1 },	// 44
	{ 1,-1,-1, 0,0,-1 },	// 45
	{ 1,0,-1, 0,0,-1 },	// 46
	{ 1,1,-1, 0,0,-1 },	// 47
};
struct Face *build_positions()
{
	return positions;
}

int disableRecord=0;
int undoLength;
int redoLength;
char undoTable[UNDOMAX];
char redoTable[UNDOMAX];
void reset_undo()
{
	undoLength=0;
	redoLength=0;
}

int undo_length()
{
	return undoLength;
}

int redo_length()
{
	return redoLength;
}

void undo(int cube[])
{
	disableRecord=1;
	if(undoLength>0) {
		undoLength--;
		int move=undoTable[undoLength];
		if((move&31)>5) {
			printf("Undo %d out of range!\n",move);
		} else if(move&32) {
			turn_cw(cube,move&31);
		} else {
			turn_ccw(cube,move&31);
		}
		redoTable[redoLength]=move;
		redoLength++;
	}
	disableRecord=0;
}

void redo(int cube[])
{
	disableRecord=1;
	if(redoLength>0) {
		redoLength--;
		int move=redoTable[redoLength];
		if((move&31)>5) {
			printf("Redo %d out of range!\n",move);
		} else if(move&32) {
			turn_ccw(cube,move&31);
		} else {
			turn_cw(cube,move&31);
		}
		undoTable[undoLength]=move;
		undoLength++;
	}
	disableRecord=0;
}

void turn_ccw( int cube[], int face)
{
    int e = face<<3, a = adj(e), n = next(e);
    int t1 = cube[e], t2 = cube[n], t3 = cube[next(a)];
    int t4 = cube[a], t5 = cube[prev(a)];

	int i;
    for( i=0; i < 3; i++) {
        int nn = next(n), nnn = next(nn);

        cube[e] = cube[nn];
        cube[n] = cube[nnn];
        cube[next(a)] = cube[adj(adj(n))];
        cube[a] = cube[adj(nn)];
        cube[prev(a)] = cube[adj(nnn)];

        e = nn;
        a = adj(e);
        n = nnn;
    }

    cube[e] = t1;
    cube[n] = t2;
    cube[next(a)] = t3;
    cube[a] = t4;
    cube[prev(a)] = t5;
    if(undoLength==UNDOMAX) {
	undoLength--;
	// keep the most recent UNDOMAX moves to undo
	memmove(undoTable,undoTable+1,UNDOMAX-1);
    }
    if(disableRecord==0) {
        undoTable[undoLength++]=face+32;
	redoLength=0;
    }
}

void turn_cw( int cube[], int face)
{
    // Let's optimize for code-size instead of speed.
    // This should be plenty fast enough.
    int oldDisable=disableRecord;
    disableRecord=1;
    turn_ccw( cube, face);
    turn_ccw( cube, face);
    turn_ccw( cube, face);
    disableRecord=oldDisable;
    if(undoLength==UNDOMAX) {
	undoLength--;
	// keep the most recent UNDOMAX moves to undo
	memmove(undoTable,undoTable+1,sizeof(undoTable));
    }
    if(disableRecord==0) {
        undoTable[undoLength++]=face;
	redoLength=0;
    }
}

int is_solved( int cube[])
{
	int i;
	for(i=0;i<48;i++) {
		// look for a face out of place.
		if(cube[i]!=i/8) return 0;
	}
	return 1;
}

struct SolverContext
{
	int pos;		    // Position in the sequence
	char sequence[200];	// the series of planned moves
    int len;
    int *tmpcube;
} solverContext;

int moves_remaining( void *context)
{
    int i,n=0;
	struct SolverContext *sc=(struct SolverContext *)context;
    for( i=sc->pos; i < sc->len; i++) {
        if( (sc->sequence[i] >> 3) == 2) n ++;
        n ++;
    }
	return n;
}

static void push_move( struct SolverContext *sc, int face, int num)
{
    if( sc->len > 0 &&
        (sc->sequence[sc->len-1] & 7) == face) {
        sc->sequence[sc->len-1] += (num << 3);
        sc->sequence[sc->len-1] &= 31;
        if((sc->sequence[sc->len-1] >> 3) == 0) {
            sc->len --;
        }
    } else {
        sc->sequence[sc->len] = face | (num << 3);
        sc->len ++;
    }
    if( num == 1) turn_cw( sc->tmpcube, face);
    else {
        turn_ccw( sc->tmpcube, face);
        if( num == 2) turn_ccw( sc->tmpcube, face);
    }
}

static int find_cubie(const int cube[], int target)
{
    int i;
    for( i=(target & 1); i < 48; i+=2) {
        if( cube[i] == face(target) &&
            cube[adj(i)] == face(adj(target))) return i;
    }
    return -1;
}

/* solves the cube from the current state
 * \param cube the cube state to start from
 * \returns context handle to change
 */
void *init_solver(const int cube[])
{
    int i, count, t, target, from, dir;
    int c[48];
    struct SolverContext *sc;
	int oldDisable=disableRecord;
	disableRecord=1;

	solverContext.pos=0;
	solverContext.len=0;
	solverContext.tmpcube=c;

    // Copy the cube
    for( i=0; i < 48; i++) c[i] = cube[i];

    sc = &solverContext;

    // Step 1: The cross
    target = 8;
    for( i=0; i < 4; i++) {
        from = find_cubie( c, target);
        if( from != target) {
            if( face(from) == 3) { // Top sticker is on the bottom
                while( face(adj(from)) != face(adj(target))) {
                    // Turn the bottom
                    push_move( sc, 3, 1);
                    from = next(next(from));
                }
                push_move( sc, face(adj(target)), 2);
            } else if( face(adj(from)) == 3) { // Adjacent to bottom
                while( face(from) != face(adj(target))) {
                    // Turn the bottom
                    push_move( sc, 3, 1);
                    from = adj(next(next(adj(from))));
                }
                push_move( sc, 3, 3);
                push_move( sc, face(adj(next(next(target)))), 3);
                push_move( sc, face(adj(target)), 1);
                push_move( sc, face(adj(next(next(target)))), 1);
            } else {
                if( face(from) == 1) {
                    push_move( sc, face(adj(from)), 1);
                    from = adj(next(next(adj(from))));
                } else if( face(adj(from)) == 1) {
                    push_move( sc, face(from), 1);
                    from = next(next(from));
                }
                count = 0;
                t = target;
                while( face(adj(t)) != face(adj(from))) {
                    push_move( sc, 1, 3);
                    t = prev(prev(t));
                    count ++;
                }

                if( face(adj(next(next(adj(from))))) == 1) {
                    push_move( sc, face(adj(from)), 1);
                } else {
                    push_move( sc, face(adj(from)), 3);
                }

                if( count) {
                    push_move( sc, 1, count);
                }
            }
        }
        target = next(next(target));
    }

ASSERT(c[8]==1 && c[10]==1 && c[12]==1 && c[14]==1);
ASSERT(c[0]==0 && c[32]==4 && c[40]==5 && c[16]==2);

    // Step 2: The top layer
    target = 9;
    for( i=0; i < 4; i++) {
        from = find_cubie( c, target);
        if( from != target) {
            if( face(from) == 1) {
                push_move( sc, face(adj(from)), 3);
                push_move( sc, 3, 1);
                push_move( sc, face(adj(from)), 1);
                from = prev(prev(adj(from)));
            } else if( face(adj(from)) == 1) {
                push_move( sc, face(from), 1);
                push_move( sc, 3, 1);
                push_move( sc, face(from), 3);
                from = prev(prev(adj(next(next(from)))));
            } else if( face(adj(adj(from))) == 1) {
                push_move( sc, face(from), 3);
                push_move( sc, 3, 3);
                push_move( sc, face(from), 1);
                from = next(next(next(next(adj(from)))));
            } else if( face(from) == 3) {
                // Move the corner under its target position
                while( face(adj(from)) != face(adj(adj(target)))) {
                    push_move( sc, 3, 1);
                    from = next(next(from));
                }
                // Now rotate it so the top color isn't on the bottom
                push_move( sc, face(adj(from)), 1);
                push_move( sc, 3, 3);
                push_move( sc, face(adj(from)), 3);
                from = adj(next(next(next(next(from)))));
            }

            // Now we know the corner cubie is in the bottom row, and
            // is not facing straight downwards.

            if( face(adj(from)) == 3) {
                while( face(from) != face(adj(target))) {
                    push_move( sc, 3, 1);
                    from = adj(prev(prev(from)));
                }
                push_move( sc, face(from), 3);
                push_move( sc, 3, 3);
                push_move( sc, face(from), 1);
            } else {
                while( face(from) != face(adj(adj(target)))) {
                    push_move( sc, 3, 1);
                    from = prev(prev(adj(from)));
                }
                push_move( sc, face(from), 1);
                push_move( sc, 3, 1);
                push_move( sc, face(from), 3);
            }
        }
        target = next(next(target));
    }

ASSERT(c[8]==1 && c[10]==1 && c[12]==1 && c[14]==1);
ASSERT(c[0]==0 && c[32]==4 && c[40]==5 && c[16]==2);
ASSERT(c[9]==1 && c[11]==1 && c[13]==1 && c[15]==1);
ASSERT(c[7]==0 && c[39]==4 && c[47]==5 && c[23]==2);

    // Step 3: The middle layer
    target = 2;
    for( i=0; i < 4; i++) {
        from = find_cubie( c, target);
        if( from != target) {
            if( face(from) != 3 && face(adj(from)) != 3) {
                // Take the cubie out of its wrong slot
                if( face(adj(next(next(from)))) == 1) from = adj(from);
                push_move( sc, face(from), 1);
                push_move( sc, 3, 1);
                push_move( sc, face(from), 3);
                push_move( sc, 3, 3);
                push_move( sc, face(adj(from)), 3);
                push_move( sc, 3, 3);
                push_move( sc, face(adj(from)), 1);
                from = find_cubie( c, target);
            }
            if( face(from) == 3) {
                while( face(adj(target)) != face(adj(prev(prev(from))))) {
                    push_move( sc, 3, 1);
                    from = next(next(from));
                }
                push_move( sc, face(target), 1);
                push_move( sc, 3, 3);
                push_move( sc, face(target), 3);
                push_move( sc, 3, 3);
                push_move( sc, face(adj(target)), 3);
                push_move( sc, 3, 1);
                push_move( sc, face(adj(target)), 1);
            } else {
                while( face(target) != face(adj(prev(prev(from))))) {
                    push_move( sc, 3, 1);
                    from = adj(next(next(adj(from))));
                }
                push_move( sc, face(adj(target)), 3);
                push_move( sc, 3, 1);
                push_move( sc, face(adj(target)), 1);
                push_move( sc, 3, 1);
                push_move( sc, face(target), 1);
                push_move( sc, 3, 3);
                push_move( sc, face(target), 3);
            }
        }
        target = next(next(next(next(adj(target)))));
    }

ASSERT(c[8]==1 && c[10]==1 && c[12]==1 && c[14]==1);
ASSERT(c[0]==0 && c[32]==4 && c[40]==5 && c[16]==2);
ASSERT(c[9]==1 && c[11]==1 && c[13]==1 && c[15]==1);
ASSERT(c[7]==0 && c[39]==4 && c[47]==5 && c[23]==2);
ASSERT(c[2]==0 && c[6]==0 && c[34]==4 && c[38]==4);
ASSERT(c[42]==5 && c[46]==5 && c[18]==2 && c[22]==2);

    // Step 4: The last layer cross
    target = -1;
    if( c[24] != 3) {
        if( c[30] != 3) target = 24;
        else if( c[26] != 3) target = 26;
    } else if( c[28] != 3) {
        if( c[30] != 3) target = 30;
        else target = 28;
    }
    if( target != -1) {
        push_move( sc, face(adj(target)), 1);
        push_move( sc, 3, 1);
        push_move( sc, face(adj(prev(prev(target)))), 1);
        push_move( sc, 3, 3);
        push_move( sc, face(adj(prev(prev(target)))), 3);
        push_move( sc, face(adj(target)), 3);
    }
    if( c[24] != 3 || c[30] != 3) {
        target = (c[24] == 3) ? 30 : 24;
        push_move( sc, face(adj(target)), 1);
        push_move( sc, face(adj(prev(prev(target)))), 1);
        push_move( sc, 3, 1);
        push_move( sc, face(adj(prev(prev(target)))), 3);
        push_move( sc, 3, 3);
        push_move( sc, face(adj(target)), 3);
    }

    // Step 5: Rotate (orient) corners
    // We sometimes need two passes
    for( i=0; i < 2; i++) {
        // How many corners are wrong?
        count = (c[25]!=3) + (c[27]!=3) + (c[29]!=3) + (c[31]!=3);
        if( count == 0) break;
        if( count == 2) {
            target = 25;
            while( c[prev(prev(target))] == 3) target = next(next(target));
            dir = (c[adj(prev(prev(target)))] == 3);
        } else if( count == 3) {
            target = 25;
            while( c[prev(prev(target))] != 3) target = next(next(target));
            dir = (c[adj(target)] != 3);
        } else {
            target = 25;
            dir = (c[35] != 3);
        }
        if( dir) {
            push_move( sc, face(adj(adj(next(next(target))))), 1);
            push_move( sc, 3, 1);
            push_move( sc, face(adj(adj(next(next(target))))), 3);
            push_move( sc, 3, 1);
            push_move( sc, face(adj(adj(next(next(target))))), 1);
            push_move( sc, 3, 2);
            push_move( sc, face(adj(adj(next(next(target))))), 3);
        } else {
            push_move( sc, face(adj(adj(target))), 3);
            push_move( sc, 3, 3);
            push_move( sc, face(adj(adj(target))), 1);
            push_move( sc, 3, 3);
            push_move( sc, face(adj(adj(target))), 3);
            push_move( sc, 3, 2);
            push_move( sc, face(adj(adj(target))), 1);
        }
    }

ASSERT(c[8]==1 && c[10]==1 && c[12]==1 && c[14]==1);
ASSERT(c[0]==0 && c[32]==4 && c[40]==5 && c[16]==2);
ASSERT(c[9]==1 && c[11]==1 && c[13]==1 && c[15]==1);
ASSERT(c[7]==0 && c[39]==4 && c[47]==5 && c[23]==2);
ASSERT(c[2]==0 && c[6]==0 && c[34]==4 && c[38]==4);
ASSERT(c[42]==5 && c[46]==5 && c[18]==2 && c[22]==2);
ASSERT(c[31]==3 && c[25]==3 && c[27]==3 && c[29]==3);

    // Step 6: Swap (permute) corners
    // Like in step 5, we sometimes need two passes
    for( i=0; i < 2; i++) {
        // Rotate the bottom until at most two corners are misplaced
        while( (count = (c[3]!=0)+(c[19]!=2)+(c[43]!=5)+(c[35]!=4)) > 2) {
            push_move( sc, 3, 1);
        }
        if( count == 0) break;
        target = 25;
        while( c[adj(adj(prev(prev(target))))] != face(adj(target)) ||
            c[adj(target)] == face(adj(target))) target = next(next(target));

        push_move( sc, face(adj(prev(prev(target)))), 1);
        push_move( sc, 3, 3);
        push_move( sc, face(adj(adj(target))), 3);
        push_move( sc, 3, 1);
        push_move( sc, face(adj(prev(prev(target)))), 3);
        push_move( sc, 3, 2);
        push_move( sc, face(adj(adj(target))), 1);
        push_move( sc, 3, 3);
        push_move( sc, face(adj(adj(target))), 3);
        push_move( sc, 3, 2);
        push_move( sc, face(adj(adj(target))), 1);
    }

ASSERT(c[8]==1 && c[10]==1 && c[12]==1 && c[14]==1);
ASSERT(c[0]==0 && c[32]==4 && c[40]==5 && c[16]==2);
ASSERT(c[9]==1 && c[11]==1 && c[13]==1 && c[15]==1);
ASSERT(c[7]==0 && c[39]==4 && c[47]==5 && c[23]==2);
ASSERT(c[2]==0 && c[6]==0 && c[34]==4 && c[38]==4);
ASSERT(c[42]==5 && c[46]==5 && c[18]==2 && c[22]==2);
ASSERT(c[31]==3 && c[25]==3 && c[27]==3 && c[29]==3);
ASSERT(c[5]==0 && c[21]==2 && c[45]==5 && c[37]==4);

    // Step 7: Carrousel (cycle) edges
    // Again, we sometimes need two passes
    for( i=0; i < 2; i++) {
        count = (c[4]!=0)+(c[20]!=2)+(c[44]!=5)+(c[36]!=4);
        if( count == 0) break;
        if( count == 4) {
            target = 24;
            dir = 0;
        } else {
            target = 24;
            while( c[adj(target)] != face(adj(target)))
                target = next(next(target));
            target = next(next(next(next(target))));
            dir = (c[adj(target)] == face(adj(next(next(target)))));
        }
        if( dir) {
            push_move( sc, face(adj(prev(prev(target)))), 1);
            push_move( sc, 3, 1);
            push_move( sc, face(adj(prev(prev(target)))), 3);
            push_move( sc, 3, 1);
            push_move( sc, face(adj(prev(prev(target)))), 1);
            push_move( sc, 3, 2);
            push_move( sc, face(adj(prev(prev(target)))), 3);
            push_move( sc, face(adj(target)), 3);
            push_move( sc, 3, 3);
            push_move( sc, face(adj(target)), 1);
            push_move( sc, 3, 3);
            push_move( sc, face(adj(target)), 3);
            push_move( sc, 3, 2);
            push_move( sc, face(adj(target)), 1);
        } else {
            push_move( sc, face(adj(next(next(target)))), 3);
            push_move( sc, 3, 3);
            push_move( sc, face(adj(next(next(target)))), 1);
            push_move( sc, 3, 3);
            push_move( sc, face(adj(next(next(target)))), 3);
            push_move( sc, 3, 2);
            push_move( sc, face(adj(next(next(target)))), 1);
            push_move( sc, face(adj(target)), 1);
            push_move( sc, 3, 1);
            push_move( sc, face(adj(target)), 3);
            push_move( sc, 3, 1);
            push_move( sc, face(adj(target)), 1);
            push_move( sc, 3, 2);
            push_move( sc, face(adj(target)), 3);
        }
    }

ASSERT(is_solved(c));
	disableRecord=oldDisable;

	return &solverContext;
}

/* returns the next move to make to solve the cube
 * \param context is the state infromation returned by init.
 * \returns face to rotate 0 to 5 for cw or 32 to 37 for ccw, or -1 if solved.
 */
int solver_next(void *context)
{
	struct SolverContext *sc=(struct SolverContext *)context;
    int face, n=0;

    if( sc->pos >= sc->len) return -1;

    while( n == 0) {
        face = sc->sequence[sc->pos] & 7;
        n = sc->sequence[sc->pos] >> 3;
        sc->pos ++;
    }

    if( n == 1) return face;
    if( n == 3) return face | 32;

    // WARNING: This is destructive to the sequence
    sc->pos --;
    sc->sequence[sc->pos] -= 8;
    return face;
}

#if 0
int main( int argc, char **argv) {
    int cube[48];
    void *solver;
    int move;

    init_cube( cube);

    // Mix up the cube
    turn_cw( cube, 1);
    turn_cw( cube, 2);
    turn_cw( cube, 3);
    turn_cw( cube, 2);
    turn_cw( cube, 1);
    turn_cw( cube, 2);
    turn_ccw( cube, 5);
    turn_cw( cube, 3);
    turn_ccw( cube, 0);
    turn_cw( cube, 3);

    solver = init_solver( cube);

    while( (move = solver_next( solver)) != -1) {
        switch( move) {
        case 0: printf("F+ "); break;
        case 1: printf("T+ "); break;
        case 2: printf("R+ "); break;
        case 3: printf("B+ "); break;
        case 4: printf("L+ "); break;
        case 5: printf("P+ "); break;
        case 32: printf("F- "); break;
        case 33: printf("T- "); break;
        case 34: printf("R- "); break;
        case 35: printf("B- "); break;
        case 36: printf("L- "); break;
        case 37: printf("P- "); break;
        default:
            printf("ERR:%d ",move); break;
        };
    }
    printf("\n");

    return 0;
}
#endif

void save_cube(int cube[])
{
	FILE *file=fopen("data/savegame","w");
	if(file) {
		fwrite(cube,48,4,file);

		fwrite(&undoLength,4,1,file);
		fwrite(&redoLength,4,1,file);

		if(undoLength) fwrite(undoTable,undoLength,1,file);
		if(redoLength) fwrite(redoTable,redoLength,1,file);

		fclose(file);
	}
}

void load_cube(int cube[])
{
	FILE *file=fopen("data/savegame","r");
	if(file) {
		fread(cube,48,4,file);

		fread(&undoLength,4,1,file);
		fread(&redoLength,4,1,file);

		if(undoLength) fread(undoTable,undoLength,1,file);
		if(redoLength) fread(redoTable,redoLength,1,file);

		fclose(file);
	}
}
